import mongoose from "mongoose";

const projectReportSchema = new mongoose.Schema(
  {
    projectId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "Project",
      required: true,
      index: true,
    },
    userId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
      index: true,
    },
    title: { type: String, required: true, trim: true },
    description: { type: String, required: true, trim: true },
    status: {
      type: String,
      enum: ["submitted", "reviewed", "approved", "needs-revision"],
      default: "submitted",
      index: true,
    },
    attachments: [
      {
        filename: String,
        url: String,
        uploadedAt: { type: Date, default: Date.now },
      },
    ],
    reviewedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
    },
    reviewNote: { type: String, trim: true },
    reviewedAt: { type: Date },
    hoursSpent: { type: Number, min: 0 },
    completionPercentage: { type: Number, min: 0, max: 100 },
  },
  { timestamps: true }
);

const userProjectStatusSchema = new mongoose.Schema(
  {
    projectId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "Project",
      required: true,
      index: true,
    },
    userId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
      index: true,
    },
    status: {
      type: String,
      enum: ["open", "in-progress", "revise", "close"],
      required: true,
      index: true,
    },
    note: { type: String, trim: true },
    updatedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
  },
  { timestamps: true }
);

userProjectStatusSchema.index({ projectId: 1, userId: 1 });

const projectSchema = new mongoose.Schema(
  {
    name: { type: String, required: true, trim: true },
    description: { type: String, trim: true },
    status: {
      type: String,
      enum: ["open", "in-progress", "revise", "close"],
      default: "open",
      index: true,
    },
    priority: {
      type: String,
      enum: ["low", "medium", "high", "urgent"],
      default: "medium",
      index: true,
    },
    createdBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    members: [
      { type: mongoose.Schema.Types.ObjectId, ref: "User", index: true },
    ],
    deadline: { type: Date },
    startDate: { type: Date },
    progress: { type: Number, min: 0, max: 100, default: 0 },
    tags: [{ type: String, trim: true }],
    channelId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "Channel",
    },
    category: {
      type: String,
      enum: [
        "graphics-design",
        "web-development",
        "ui-ux-design",
        "digital-marketing",
        "video-production",
        "general",
      ],
      default: "general",
      index: true,
    },
  },
  { timestamps: true }
);

export const ProjectReport = mongoose.model(
  "ProjectReport",
  projectReportSchema
);
export const UserProjectStatus = mongoose.model(
  "UserProjectStatus",
  userProjectStatusSchema
);
export const Project = mongoose.model("Project", projectSchema);
