import { z } from "zod";

export const createChannelSchema = z.object({
  body: z.object({
    name: z.string().min(2),
    description: z.string().max(500).optional(),
    type: z
      .enum(["general", "project", "team", "announcement", "private", "group"])
      .optional(),
    groupType: z
      .enum([
        "employee-only",
        "employee-intern",
        "employee-student",
        "all-staff",
      ])
      .optional(),
    allowedRoles: z
      .array(z.enum(["admin", "employee", "intern", "student", "customer"]))
      .optional(),
    category: z
      .enum([
        "graphics-design",
        "web-development",
        "ui-ux-design",
        "digital-marketing",
        "video-production",
        "general",
      ])
      .optional(),
    isPrivate: z
      .union([z.boolean(), z.string()])
      .optional()
      .transform((val) => {
        if (typeof val === "boolean") return val;
        if (typeof val === "string") return val === "true" || val === "1";
        return false;
      }),
    memberIds: z.array(z.string().min(1)).optional(),
  }),
});

export const updateChannelSchema = z.object({
  params: z.object({ channelId: z.string().min(1) }),
  body: z.object({
    name: z.string().min(2).optional(),
    description: z.string().max(500).optional(),
    memberIds: z.array(z.string().min(1)).optional(),
    isPinned: z.boolean().optional(),
  }),
});

export const postMessageSchema = z.object({
  params: z.object({ channelId: z.string().min(1) }),
  body: z
    .object({
      text: z.string().max(5000).optional(),
      replyTo: z.string().optional(),
      attachments: z
        .array(
          z.object({
            filename: z.string().min(1),
            url: z.string().min(1),
            type: z.enum(["image", "file"]).optional(),
          })
        )
        .optional(),
    })
    .refine(
      (val) => {
        const hasText =
          typeof val.text === "string" && val.text.trim().length > 0;
        const hasAttachments =
          Array.isArray(val.attachments) && val.attachments.length > 0;
        return hasText || hasAttachments;
      },
      { message: "Message must include text or at least one attachment" }
    ),
});
