import { z } from "zod";

const userRoleSchema = z.enum(["employee", "intern", "student", "customer"]);

const userCategorySchema = z.enum([
  "graphics-design",
  "web-development",
  "ui-ux-design",
  "digital-marketing",
  "video-production",
  "general",
]);

export const loginSchema = z.object({
  body: z
    .object({
      email: z.string().email().optional(),
      id: z.string().min(1).optional(),
      role: userRoleSchema.optional(),
      password: z.string().min(6),
    })
    .refine((b) => Boolean(b.email || b.id), {
      message: "email or id is required",
    }),
});

export const seedAdminSchema = z.object({
  body: z.object({
    fullName: z.string().min(2),
    email: z.string().email(),
    password: z.string().min(10),
  }),
});

export const registerSchema = z.object({
  body: z
    .object({
      fullName: z.string().min(2),
      email: z.string().email(),
      phone: z.string().min(7).max(25),
      password: z.string().min(8),
      role: userRoleSchema.optional(),
      category: userCategorySchema.optional(),
    })
    .refine(
      (b) => {
        const role = b.role ?? "employee";
        if (role === "intern" || role === "student") {
          return Boolean(b.category);
        }
        return true;
      },
      { message: "category is required for intern/student registration" }
    ),
});

export const forgotPasswordSchema = z.object({
  body: z.object({
    email: z.string().email(),
  }),
});

export const resetPasswordSchema = z.object({
  body: z.object({
    email: z.string().email(),
    token: z.string().min(1),
    newPassword: z.string().min(8),
  }),
});

export const changePasswordSchema = z.object({
  body: z.object({
    currentPassword: z.string().min(1),
    newPassword: z.string().min(8),
  }),
});
